/**
* \file: message_buffer.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AMS_MESSAGE_BUFFER_H
#define AMS_MESSAGE_BUFFER_H

#include "utils/automounter_types_internal.h"

typedef enum message_buffer_state_t
{
	MSG_BUFFER_EMPTY,
	MSG_BUFFER_SENDING,
	MSG_BUFFER_RECEIVING,
	MSG_BUFFER_MSG_SENT,
	MSG_BUFFER_MSG_RECEIVED,
	MSG_BUFFER_MSG_PREPARED_FOR_SENDING
} message_buffer_state_t;

typedef struct message_buffer_t message_buffer_t;

typedef struct message_queue_t
{
	message_buffer_t *first_buffer;
	message_buffer_t *last_buffer;
	unsigned int no_elements;
} message_queue_t;

//------------------------------------- public member - buffer mgnt -------------------------------------------------
/**
 * creates a new buffer with initial buffer size which is in state MSG_BUFFER_EMPTY
 */
message_buffer_t *message_buffer_create_new(size_t initial_size);

/**
 * This function removes the given buffer from memory.
 */
void message_buffer_destroy(message_buffer_t *msg_buffer);

/**
 * This function puts the buffer into state MSG_BUFFER_EMPTY
 */
void message_buffer_mark_empty(message_buffer_t *msg_buffer);

/**
 * This function can be used to send the message in the buffer again. This function requires that
 * the buffer is in state MSG_BUFFER_MSG_SENT. Internally, it resets the "bytes processed" counter and changes the
 * buffer state to MSG_BUFFER_MSG_PREPARED_FOR_SENDING. This allows using the message_buffer_send() function with this
 * buffer again without creating the message again. Normally, this member is used to broadcast messages.
 */
void message_buffer_reuse_message_for_sending(message_buffer_t *msg_buffer);
//-------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - queueing ----------------------------------------------------
/**
 * call to initialize a new message queue data structure
 */
void message_buffer_init_queue(message_queue_t *queue);

/**
 * inserts the given buffer at the beginning of the queue. Be careful, the queue is a linked list. a message buffer can
 * be linked only in one list at a time. A buffer that is already linked into another list is not inserted.
 */
void message_buffer_insert_at_begining(message_queue_t *queue, message_buffer_t *message_buffer);

/**
 * deuques the first buffer from the queue and returns it. If the queue is empty, NULL is returned.
 */
message_buffer_t  *message_buffer_deque_from_begining(message_queue_t *queue);

/**
 * removes the buffer from the queue it is linked in.
 */
void message_buffer_remove_from_queue(message_queue_t *queue, message_buffer_t *message_buffer);
//-------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - sending -----------------------------------------------------
/**
 * This member prepares tlhe given buffer for taking a new message.
 * It is expected that the given buffer is in state MSG_BUFFER_EMPTY. The message type of the new message is set
 * according to the given parameter. A fixed amount of memory of size data_size is reservered for the new message and
 * the given ptr is set to the memory area, where the data can be stored.
 */
error_code_t message_buffer_prepare_message(message_buffer_t *msg_buffer, message_type_t message_type,
		size_t data_size, void **data_buffer);

/**
 * This member adds a string to a message prepared in the given buffer.
 * This member is valid only if the buffer is in state MSG_BUFFER_MSG_PREPARED_FOR_SENDING.
 */
error_code_t message_buffer_add_string(message_buffer_t *msg_buffer, const char *string);

/**
 * This member sends the message in the buffer to the given socket.
 * It tries to send the complete message with one call. If the message can't be send completely, the buffer enters the
 * state MSG_BUFFER_SENDING. A subsequent call of this function tries to send the rest of the message. Once the message
 * has been sent completely, the buffer automatically enters the state MSG_BUFFER_EMPTY.
 * A call to this function is valid only if the buffer is in state MSG_BUFFER_SENDING or in
 * state MSG_BUFFER_MSG_PREPARED_FOR_SENDING.
 */
error_code_t message_buffer_send(message_buffer_t *msg_buffer, int a_socket);

/**
 * this function clones the message currently stored in src_buffer. src_buffer needs to be in state MSG_BUFFER_SENDING
 * or in state MSG_BUFFER_PREPARED_FOR_SENDING. The dst_buffer needs to be in state MSG_BUFFER_EMPTY. After cloning
 * successfully the message, the dst_buffer will be in state MSG_PREPARED_FOR_SENDING.
 *
 * This function is used to send broadcast messages. In case the message couldn't be sent completely to one of the
 * receivers, a copy of this one can be taken and used to send the message to other ones while the first buffer is
 * queued into the sender queue.
 */
error_code_t message_buffer_clone_msg_in_buffer(message_buffer_t *dst_buffer, message_buffer_t *src_buffer);

//--------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - receiving ----------------------------------------------------

/**
 * This member can be used to receive a message from the given socket and put it into the given buffer.
 * A call to this member is only valid for a buffer that is either in state MSG_BUFFER_EMPTY or MSG_BUFFER_RECEIVING.
 * The member tries to read a complete message. If not a complete message can be read, it reads only the data it
 * can get and puts it into the buffer. A second call to this function for the same buffer goes on reading until the
 * message is complete. It is expected that the data read from the socket belongs to one message even if more than one
 * call is needed to read out the data.
 */
error_code_t message_buffer_receive(message_buffer_t *msg_buffer, int a_socket);

/**
 * This call returns the message type of a message in the buffer.
 * This call is valid only if the given buffer is in state MSG_BUFFER_MSG_RECEIVED or
 * MSG_BUFFER_MSG_PREPARED_FOR_SENDING. In all other states, the function returns __NO_MSG.
 */
message_type_t message_buffer_get_msg_type(message_buffer_t *msg_buffer);

/**
 * This call set the given buf_ptr to the data part in the message contained in the given buffer.
 * This call is valid only if the given buffer is in state MSG_BUFFER_MSG_RECEIVED or
 * MSG_BUFFER_MSG_PREPARED_FOR_SENDING. In all other states, the function set the buf_ptr to NULL.
 */
void message_buffer_get_msg_data(message_buffer_t *msg_buffer, void **buf_ptr, size_t *buffer_size_ptr);

/**
 * This call returns the first string of the message contained in the given buffer.
 * This call is valid only if the given buffer is in state MSG_BUFFER_MSG_RECEIVED or
 * MSG_BUFFER_MSG_PREPARED_FOR_SENDING. In all other states, the function returns NULL.
 */
const char *message_buffer_get_first_msg_string(message_buffer_t *msg_buffer);

/**
 * This call returns the next string of the message contained in the given buffer.
 * If no other strings are contained in the message anymore, it returns NULL. The internal iterator is reset
 * on a call of message_buffer_get_first_msg_string().
 * This call is valid only if the given buffer is in state MSG_BUFFER_MSG_RECEIVED or
 * MSG_BUFFER_MSG_PREPARED_FOR_SENDING. In all other states, the function returns NULL.
 */
const char *message_buffer_get_next_msg_string(message_buffer_t *msg_buffer);
//--------------------------------------------------------------------------------------------------------------------

#endif
